# yolov8_detector.py
import cv2
import numpy as np
from ultralytics import YOLO
import threading
import torch

class YOLOv8Detector:
    def __init__(self, model_path: str, conf_threshold: float = 0.25, iou_threshold: float = 0.45):
        self.model_path = model_path
        self.conf_threshold = conf_threshold
        self.iou_threshold = iou_threshold
        
        
        self.lock = threading.Lock()
        
        # تحميل النموذج
        self._load_model()

    def _load_model(self):

        try:
            self.model = YOLO(self.model_path)
            # تحسين أداء النموذج للاستدلال
            self.model.fuse() 
            print(f"YOLOv8 model loaded from {self.model_path}")
        except Exception as e:
            print(f"Error loading YOLOv8 model: {e}")
            raise

    def detect_image(self, image_np: np.ndarray) -> list:

        if not hasattr(self, 'model'):
            print("Model is not loaded.")
            return []

        try:
            with self.lock:
             
                results = self.model(
                    image_np,
                    conf=self.conf_threshold,
                    iou=self.iou_threshold,
                    imgsz=640,           
                    device=0 if torch.cuda.is_available() else 'cpu',
                    verbose=False  
                )
            
            
            result = results[0]
            
            detections = []
            for box in result.boxes:
           
                bbox = box.xyxy[0].cpu().numpy().astype(int).tolist()
                
              
                score = float(box.conf[0].cpu().numpy())
                class_id = int(box.cls[0].cpu().numpy())
                class_name = self.model.names[class_id]
                
                detections.append({
                    "bbox": bbox,
                    "score": score,
                    "class_id": class_id,
                    "class_name": class_name
                })
            
            print(f"Detected {len(detections)} objects")
            return detections

        except Exception as e:
            print(f"Error during YOLOv8 detection: {str(e)}")
            return []





