# # depth_estimator.py
# import cv2
# import torch
# import numpy as np
# import threading

# class MiDaSEstimator:
#     def __init__(self, model_type: str = "MiDaS_small"):
#         """
#         يقوم بتحميل نموذج MiDaS لتقدير العمق.
#         'MiDaS_small' هو الأسرع والأقل استهلاكاً للموارد.
#         """
#         self.model_type = model_type
#         self.lock = threading.Lock()
#         self._load_model()

#     def _load_model(self):
#         """
#         يقوم بتحميل النموذج والمحول (transform) من PyTorch Hub.
#         """
#         try:
#             self.device = torch.device("cuda" if torch.cuda.is_available() else "cpu")
#             print(f"MiDaS using device: {self.device}")

#             self.model = torch.hub.load("intel-isl/MiDaS", self.model_type)
#             self.model.to(self.device)
#             self.model.eval()

#             transforms = torch.hub.load("intel-isl/MiDaS", "transforms")
#             self.transform = transforms.dpt_transform if "DPT" in self.model_type else transforms.small_transform
            
#             print(f"✅ MiDaS model '{self.model_type}' loaded successfully.")
#         except Exception as e:
#             print(f"❌ Error loading MiDaS model: {e}")
#             raise

#     def estimate_depth(self, image_np: np.ndarray) -> np.ndarray:
#         """
#         يقدر العمق في صورة ويعيد خريطة العمق.
#         """
#         if not hasattr(self, 'model'):
#             print("MiDaS model is not loaded.")
#             return None

#         try:
#             img_rgb = cv2.cvtColor(image_np, cv2.COLOR_BGR2RGB)
#             input_batch = self.transform(img_rgb).to(self.device)

#             with torch.no_grad(), self.lock:
#                 prediction = self.model(input_batch)
#                 prediction = torch.nn.functional.interpolate(
#                     prediction.unsqueeze(1),
#                     size=img_rgb.shape[:2],
#                     mode="bicubic",
#                     align_corners=False,
#                 ).squeeze()

#             depth_map = prediction.cpu().numpy()
#             return depth_map

#         except Exception as e:
#             print(f"Error during MiDaS depth estimation: {str(e)}")
#             return None


# # depth_estimator.py
# import torch
# import cv2
# import numpy as np
# import threading
# from transformers import AutoImageProcessor, AutoModelForDepthEstimation

# class DepthAnythingEstimator:
#     def __init__(self, model_name: str = "LiheYoung/depth-anything-small-hf"):
#         """
#         يقوم بتحميل نموذج DepthAnything لتقدير العمق.
#         خيارات أخرى للاسم:
#         - "LiheYoung/depth-anything-base-hf" (أكبر وأدق)
#         - "LiheYoung/depth-anything-large-hf" (الأكبر والأكثر دقة)
#         نبدأ بالنموذج الصغير لأنه الأسرع.
#         """
#         self.model_name = model_name
#         self.lock = threading.Lock()
#         self._load_model()

#     def _load_model(self):
#         """
#         يقوم بتحميل النموذج والمعالج (processor) من Hugging Face.
#         """
#         try:
#             self.device = torch.device("cuda" if torch.cuda.is_available() else "cpu")
#             print(f"DepthAnything using device: {self.device}")

#             # تحميل المعالج والنموذج
#             self.image_processor = AutoImageProcessor.from_pretrained(self.model_name)
#             self.model = AutoModelForDepthEstimation.from_pretrained(self.model_name)
#             self.model.to(self.device)
#             self.model.eval()

#             print(f"✅ DepthAnything model '{self.model_name}' loaded successfully.")
#         except Exception as e:
#             print(f"❌ Error loading DepthAnything model: {e}")
#             raise

#     def estimate_depth(self, image_np: np.ndarray) -> np.ndarray:
#         """
#         يقدر العمق في صورة ويعيد خريطة العمق.
#         """
#         if not hasattr(self, 'model'):
#             print("DepthAnything model is not loaded.")
#             return None

#         try:
#             # تحويل الصورة من BGR (OpenCV) إلى RGB
#             image_rgb = cv2.cvtColor(image_np, cv2.COLOR_BGR2RGB)

#             # معالجة الصورة لتناسب النموذج
#             inputs = self.image_processor(images=image_rgb, return_tensors="pt").to(self.device)

#             with torch.no_grad(), self.lock:
#                 # إجراء التنبؤ بالعمق
#                 outputs = self.model(**inputs)
#                 predicted_depth = outputs.predicted_depth

#                 # تغيير حجم خريطة العمق لتطابق حجم الصورة الأصلية
#                 prediction = torch.nn.functional.interpolate(
#                     predicted_depth.unsqueeze(1),
#                     size=image_rgb.shape[:2],
#                     mode="bicubic",
#                     align_corners=False,
#                 ).squeeze()

#             # نقل النتيجة إلى الـ CPU وتحويلها إلى numpy array
#             depth_map = prediction.cpu().numpy()
#             return depth_map

#         except Exception as e:
#             print(f"Error during DepthAnything depth estimation: {str(e)}")
#             return None



# depth_estimator.py
import torch
import cv2
import numpy as np
import threading
from transformers import AutoImageProcessor, AutoModelForDepthEstimation

class DepthAnythingEstimator:
    def __init__(self, model_name: str = "LiheYoung/depth-anything-small-hf"):

        self.model_name = model_name
        self.lock = threading.Lock()
        self._load_model()

    def _load_model(self):

        try:
            self.device = torch.device("cuda" if torch.cuda.is_available() else "cpu")
            print(f"DepthAnything using device: {self.device}")

            self.image_processor = AutoImageProcessor.from_pretrained(self.model_name)
            self.model = AutoModelForDepthEstimation.from_pretrained(self.model_name)
            
            self.model.to(self.device)
            self.model.eval()

            print(f" DepthAnything model '{self.model_name}' loaded successfully.")
        except Exception as e:
            print(f" Error loading DepthAnything model: {e}")
            raise

    def estimate_depth(self, image_np: np.ndarray) -> np.ndarray:

        if not hasattr(self, 'model'):
            return None
        try:
            image_rgb = cv2.cvtColor(image_np, cv2.COLOR_BGR2RGB)
            inputs = self.image_processor(images=image_rgb, return_tensors="pt").to(self.device)

            with torch.no_grad(), self.lock:
                outputs = self.model(**inputs)
                predicted_depth = outputs.predicted_depth
                prediction = torch.nn.functional.interpolate(
                    predicted_depth.unsqueeze(1),
                    size=image_rgb.shape[:2],
                    mode="bicubic",
                    align_corners=False,
                ).squeeze()
            
            depth_map = prediction.cpu().numpy()
            return depth_map
        except Exception as e:
            print(f"Error during DepthAnything depth estimation: {str(e)}")
            return None
