from ultralytics import YOLO
import os
import subprocess
import tensorflow as tf

def export_to_onnx(pt_model_path):
    print("[1] Exporting to ONNX...")
    model = YOLO(pt_model_path)
    # تصدير النموذج إلى ONNX مع opset 16 (يمكنك تغييره حسب الحاجة)
    model.export(format="onnx", opset=16, simplify=True, dynamic=False)
    onnx_path = os.path.splitext(pt_model_path)[0] + ".onnx"
    print(f"✔ ONNX exported: {onnx_path}")
    return onnx_path

def convert_onnx_to_savedmodel(onnx_model_path, saved_model_dir):
    print("[2] Converting ONNX to TensorFlow SavedModel using onnx-tf...")
    command = [
        "onnx-tf",
        "convert",
        "-i", onnx_model_path,
        "-o", saved_model_dir
    ]
    result = subprocess.run(command, capture_output=True, text=True)
    if result.returncode != 0:
        print("Error during ONNX to SavedModel conversion:")
        print(result.stderr)
        raise RuntimeError("ONNX to SavedModel conversion failed.")
    print(f"✔ SavedModel created: {saved_model_dir}")

def convert_to_tflite(saved_model_dir, tflite_file_path):
    print("[3] Converting SavedModel to TFLite with quantization...")

    converter = tf.lite.TFLiteConverter.from_saved_model(saved_model_dir)

    # تفعيل دعم ops الأساسية وإضافة دعم ops من TensorFlow في حالة وجود بعض العمليات الخاصة
    converter.target_spec.supported_ops = [
        tf.lite.OpsSet.TFLITE_BUILTINS,    # عمليات TFLite الأساسية
        tf.lite.OpsSet.SELECT_TF_OPS       # عمليات TensorFlow الاختيارية
    ]

    # تفعيل الكمّية لتقليل حجم النموذج وتسريع الأداء
    converter.optimizations = [tf.lite.Optimize.DEFAULT]

    # يمكنك إضافة quantization input/output type (اختياري)
    # converter.inference_input_type = tf.uint8
    # converter.inference_output_type = tf.uint8

    tflite_model = converter.convert()
    with open(tflite_file_path, "wb") as f:
        f.write(tflite_model)
    print(f"✔ TFLite model saved: {tflite_file_path}")

def main():
    pt_model = "C:/Users/HP/Desktop/Basiraty app/object_detection/best (yolov8m).pt"
    saved_model_dir = "C:/Users/HP/Desktop/Basiraty app/object_detection/converted/saved_model"
    tflite_file_path = "C:/Users/HP/Desktop/Basiraty app/object_detection/converted/model_quantized.tflite"

    onnx_model_path = export_to_onnx(pt_model)
    convert_onnx_to_savedmodel(onnx_model_path, saved_model_dir)
    convert_to_tflite(saved_model_dir, tflite_file_path)

if __name__ == "__main__":
    main()
